;***************************************************************
;
; File: BOOT.ASM
;
; Description:
;
;       This is the code for the DSP56002EVM which allows an
;       application to be programmed into the FLASH EEPROM
;       and booted from the FLASH EEPROM.
;       See the file BOOT.TXT for information on how to use
;       this code.
; 
; Revisions:
;
;    28 JUN 95  CMH  1. Created.
;    30 JUN 95  CMH  1. Updated.
;
;***************************************************************
        opt     cex,mex
        page    132,66,0,0
        nolist
        list

TRUE_           equ     1
FALSE_          equ     0
EEPROM          equ     $8000
FLASH_DELAY     equ     4000
TOGGLE_BIT      equ     6
BEGIN           equ     $7C00

RAM_P_INT       equ     $0000
RAM_X_INT       equ     $0000
RAM_Y_INT       equ     $0000
RAM_P_EXT       equ     $0200
RAM_X_EXT       equ     $0800
RAM_Y_EXT       equ     $2000

EEPROM_P_INT    equ     $C600
EEPROM_X_INT    equ     $CC00
EEPROM_Y_INT    equ     $CF00
EEPROM_P_EXT    equ     $E000
EEPROM_X_EXT    equ     $8000
EEPROM_Y_EXT    equ     $D200
EEPROM_BOOT     equ     $C000

BLOCKS_P_INT    equ     8
BLOCKS_X_INT    equ     4
BLOCKS_Y_INT    equ     4
BLOCKS_P_EXT    equ     40
BLOCKS_X_EXT    equ     80
BLOCKS_Y_EXT    equ     16
BLOCKS_BOOT     equ     8

PLL             equ     $FFFD
BCR             equ     $FFFE
INIT_PLL        equ     $261009
INIT_BCR        equ     $BBB0
BOOT_RUN        equ     $0000
LOAD_RUN        equ     $7D00
START           equ     $0000

        org     p:BEGIN

;***************************************************************
;
;  FlashCopy
;
;  This routine copies P, X, and Y RAM to the FLASH EEPROM.
;  This routine also copies the boot and loader code to
;  the EEPROM at P:$C000 so that the DSP boot loader will
;  copy it into P RAM and execute it.
;
; Inputs:
;
; Outputs:
;
; Registers modified:
;
;       a, r0, r1, r2
;
;***************************************************************
FlashCopy

        move    #CopyP,r0       ;return address
        jmp     FlashInit       ;init DSP registers

; Copy P memory
CopyP
        move    #RAM_P_INT,r0           ;get start address of P RAM
        move    #EEPROM_P_INT,r1        ;get start address of P EEPROM
        move    #BLOCKS_P_INT,r2        ;get number of 64 word blocks to copy
        jsr     FlashWriteP     ;copy P RAM to FLASH EEPROM
        move    #RAM_P_EXT,r0           ;get start address of P RAM
        move    #EEPROM_P_EXT,r1        ;get start address of P EEPROM
        move    #BLOCKS_P_EXT,r2        ;get number of 64 word blocks to copy
        jsr     FlashWriteP     ;copy P RAM to FLASH EEPROM

; Copy X memory
CopyX
        move    #RAM_X_INT,r0           ;get start address of X RAM
        move    #EEPROM_X_INT,r1        ;get start address of X EEPROM
        move    #BLOCKS_X_INT,r2        ;get number of 64 word blocks to copy
        jsr     FlashWriteX     ;copy X RAM to FLASH EEPROM
        move    #RAM_X_EXT,r0           ;get start address of X RAM
        move    #EEPROM_X_EXT,r1        ;get start address of X EEPROM
        move    #BLOCKS_X_EXT,r2        ;get number of 64 word blocks to copy
        jsr     FlashWriteX     ;copy X RAM to FLASH EEPROM

; Copy Y memory
CopyY
        move    #RAM_Y_INT,r0           ;get start address of Y RAM
        move    #EEPROM_Y_INT,r1        ;get start address of Y EEPROM
        move    #BLOCKS_Y_INT,r2        ;get number of 64 word blocks to copy
        jsr     FlashWriteY     ;copy Y RAM to FLASH EEPROM
        move    #RAM_Y_EXT,r0           ;get start address of Y RAM
        move    #EEPROM_Y_EXT,r1        ;get start address of Y EEPROM
        move    #BLOCKS_Y_EXT,r2        ;get number of 64 word blocks to copy
        jsr     FlashWriteY     ;copy Y RAM to FLASH EEPROM

; Copy boot and loader code to EEPROM
CopyBoot
        move    #BOOT_LOAD,r0   ;get load address
        move    #EEPROM_BOOT,r1 ;get boot address
        move    #BLOCKS_BOOT,r2 ;get number of 64 word blocks to copy
        jsr     FlashWriteP     ;copy P RAM to FLASH EEPROM

FlashCopyEnd
        jmp     *

;***************************************************************
;
;  FlashWriteP
;
;  This routine writes blocks of 64 24-bit words (192 bytes) from
;  the P RAM to the FLASH EEPROM.
;
; Inputs:
;
;       r0 : Starting address of the P RAM where the
;            blocks of words are read from
;       r1 : Starting address of the FLASH EEPROM where the
;            blocks of words are to be written
;       r2 : Number of blocks to write
;
; Outputs:
;
; Registers modified:
;
;       a, r0, r1
;
;***************************************************************
FlashWriteP

        do      r2,LoopWriteP   ;loop for number of blocks

;***************************************************************
; Do block 1. This consists of:
;             words 1 to 21 +
;             low byte of word 22
;***************************************************************

        jsr     FlashEnable     ;enable EEPROM writes

        do      #21,MoveBlockP1 ;move 21 words
        move    p:(r0)+,a       ;get words 1 to 21
        jsr     WriteWord       ;write bytes to FLASH EEPROM
MoveBlockP1
        move    p:(r0)+,a       ;get 22nd word
        move    a1,p:(r1)+      ;move low byte to FLASH EEPROM

        jsr     FlashDelay      ;delay for programming cycle to begin
        jsr     FlashCheck      ;check if programming cycle complete

;***************************************************************
; Do block 2. This consists of:
;             mid and high bytes of word 22 +
;             words 23 to 42 +
;             low and mid bytes of word 43
;***************************************************************

        jsr     FlashEnable     ;enable EEPROM writes

        rep     #8              ;continue with 22nd word
        lsr     a
        move    a1,p:(r1)+      ;move mid byte to FLASH EEPROM
        rep     #8
        lsr     a
        move    a1,p:(r1)+      ;move high byte to FLASH EEPROM
        do      #20,MoveBlockP2 ;move 20 words
        move    p:(r0)+,a       ;get words 23 to 42
        jsr     WriteWord       ;write bytes to FLASH EEPROM
MoveBlockP2
        move    p:(r0)+,a       ;get 43rd word
        move    a1,p:(r1)+      ;move low byte to FLASH EEPROM
        rep     #8
        lsr     a
        move    a1,p:(r1)+      ;move mid byte to FLASH EEPROM

        jsr     FlashDelay      ;delay for programming cycle to begin
        jsr     FlashCheck      ;check if programming cycle complete

;***************************************************************
; Do block 3. This consists of:
;             high byte of word 43 +
;             words 44 to 64
;***************************************************************

        jsr     FlashEnable     ;enable EEPROM writes

        rep     #8              ;continue with 43rd word
        lsr     a
        move    a1,p:(r1)+      ;move high byte to FLASH EEPROM
        do      #21,MoveBlockP3 ;move 21 words
        move    p:(r0)+,a       ;get words 44 to 64
        jsr     WriteWord       ;write bytes to FLASH EEPROM
MoveBlockP3

        jsr     FlashDelay      ;delay for programming cycle to begin
        jsr     FlashCheck      ;check if programming cycle complete

        jsr     FlashDisable    ;disable EEPROM writes

LoopWriteP
        nop

FlashWritePEnd
        rts

;***************************************************************
;
;  FlashWriteX
;
;  This routine writes blocks of 64 24-bit words (192 bytes) from
;  the X RAM to the FLASH EEPROM.
;
; Inputs:
;
;       r0 : Starting address of the X RAM where the
;            blocks of words are read from
;       r1 : Starting address of the FLASH EEPROM where the
;            blocks of words are to be written
;       r2 : Number of blocks to write
;
; Outputs:
;
; Registers modified:
;
;       a, r0, r1
;
;***************************************************************
FlashWriteX

        do      r2,LoopWriteX   ;loop for number of blocks

;***************************************************************
; Do block 1. This consists of:
;             words 1 to 21 +
;             low byte of word 22
;***************************************************************

        jsr     FlashEnable     ;enable EEPROM writes

        do      #21,MoveBlockX1 ;move 21 words
        move    x:(r0)+,a       ;get words 1 to 22
        jsr     WriteWord       ;write bytes to FLASH EEPROM
MoveBlockX1
        move    x:(r0)+,a       ;get 22nd word
        move    a1,p:(r1)+      ;move low byte to FLASH EEPROM

        jsr     FlashDelay      ;delay for programming cycle to begin
        jsr     FlashCheck      ;check if programming cycle complete

;***************************************************************
; Do block 2. This consists of:
;             mid and high bytes of word 22 +
;             words 23 to 42 +
;             low and mid bytes of word 43
;***************************************************************

        jsr     FlashEnable     ;enable EEPROM writes

        rep     #8              ;continue with 22nd word
        lsr     a
        move    a1,p:(r1)+      ;move mid byte to FLASH EEPROM
        rep     #8
        lsr     a
        move    a1,p:(r1)+      ;move high byte to FLASH EEPROM
        do      #20,MoveBlockX2 ;move 20 words
        move    x:(r0)+,a       ;get words 23 to 42
        jsr     WriteWord       ;write bytes to FLASH EEPROM
MoveBlockX2
        move    x:(r0)+,a       ;get 43rd word
        move    a1,p:(r1)+      ;move low byte to FLASH EEPROM
        rep     #8
        lsr     a
        move    a1,p:(r1)+      ;move mid byte to FLASH EEPROM

        jsr     FlashDelay      ;delay for programming cycle to begin
        jsr     FlashCheck      ;check if programming cycle complete

;***************************************************************
; Do block 3. This consists of:
;             high byte of word 43 +
;             words 44 to 64
;***************************************************************

        jsr     FlashEnable     ;enable EEPROM writes

        rep     #8              ;continue with 43rd word
        lsr     a
        move    a1,p:(r1)+      ;move high byte to FLASH EEPROM
        do      #21,MoveBlockX3 ;move 21 words
        move    x:(r0)+,a       ;get words 44 to 64
        jsr     WriteWord       ;write bytes to FLASH EEPROM
MoveBlockX3

        jsr     FlashDelay      ;delay for programming cycle to begin
        jsr     FlashCheck      ;check if programming cycle complete

        jsr     FlashDisable    ;disable EEPROM writes

LoopWriteX
        nop

FlashWriteXEnd
        rts

;***************************************************************
;
;  FlashWriteY
;
;  This routine writes blocks of 64 24-bit words (192 bytes) from
;  the Y RAM to the FLASH EEPROM.
;
; Inputs:
;
;       r0 : Starting address of the Y RAM where the
;            blocks of words are read from
;       r1 : Starting address of the FLASH EEPROM where the
;            blocks of words are to be written
;       r2 : Number of blocks to write
;
; Outputs:
;
; Registers modified:
;
;       a, r0, r1
;
;***************************************************************
FlashWriteY

        do      r2,LoopWriteY   ;loop for number of blocks

;***************************************************************
; Do block 1. This consists of:
;             words 1 to 21 +
;             low byte of word 22
;***************************************************************

        jsr     FlashEnable     ;enable EEPROM writes

        do      #21,MoveBlockY1 ;move 21 words
        move    y:(r0)+,a       ;get words 1 to 21
        jsr     WriteWord       ;write bytes to FLASH EEPROM
MoveBlockY1
        move    y:(r0)+,a       ;get 22nd word
        move    a1,p:(r1)+      ;move low byte to FLASH EEPROM

        jsr     FlashDelay      ;delay for programming cycle to begin
        jsr     FlashCheck      ;check if programming cycle complete

;***************************************************************
; Do block 2. This consists of:
;             mid and high bytes of word 22 +
;             words 23 to 42 +
;             low and mid bytes of word 43
;***************************************************************

        jsr     FlashEnable     ;enable EEPROM writes

        rep     #8              ;continue with 22nd word
        lsr     a
        move    a1,p:(r1)+      ;move mid byte to FLASH EEPROM
        rep     #8
        lsr     a
        move    a1,p:(r1)+      ;move high byte to FLASH EEPROM
        do      #20,MoveBlockY2 ;move 20 words
        move    y:(r0)+,a       ;get words 23 to 42
        jsr     WriteWord       ;write bytes to FLASH EEPROM
MoveBlockY2
        move    y:(r0)+,a       ;get 43rd word
        move    a1,p:(r1)+      ;move low byte to FLASH EEPROM
        rep     #8
        lsr     a
        move    a1,p:(r1)+      ;move mid byte to FLASH EEPROM

        jsr     FlashDelay      ;delay for programming cycle to begin
        jsr     FlashCheck      ;check if programming cycle complete

;***************************************************************
; Do block 3. This consists of:
;             high byte of word 43 +
;             words 44 to 64
;***************************************************************

        jsr     FlashEnable     ;enable EEPROM writes

        rep     #8              ;continue with 43rd word
        lsr     a
        move    a1,p:(r1)+      ;move high byte to FLASH EEPROM
        do      #21,MoveBlockY3 ;move 21 words
        move    y:(r0)+,a       ;get words 44 to 64
        jsr     WriteWord       ;write bytes to FLASH EEPROM
MoveBlockY3

        jsr     FlashDelay      ;delay for programming cycle to begin
        jsr     FlashCheck      ;check if programming cycle complete

        jsr     FlashDisable    ;disable EEPROM writes

LoopWriteY
        nop

FlashWriteYEnd
        rts

;***************************************************************
;
;  FlashDisable
;
;  This function performs the sequence for activating software
;  data protection, i.e. disables writes to the EEPROM.
;
;***************************************************************
FlashDisable
        move    #>$AA,x0
        move    x0,x:EEPROM+$5555
        move    #>$55,x0
        move    x0,x:EEPROM+$2AAA
        move    #>$A0,x0
        move    x0,x:EEPROM+$5555
        rts                     ;writes are now disabled
           
;***************************************************************
;
;  FlashEnable
;
;  This function performs the sequence for deactivating software
;  data protection, i.e. enables writes to the EEPROM.
;
;***************************************************************
FlashEnable
        move    #>$AA,x0
        move    x0,x:EEPROM+$5555
        move    #>$55,x0
        move    x0,x:EEPROM+$2AAA
        move    #>$80,x0
        move    x0,x:EEPROM+$5555
        move    #>$AA,x0
        move    x0,x:EEPROM+$5555
        move    #>$55,x0
        move    x0,x:EEPROM+$2AAA
        move    #>$20,x0
        move    x0,x:EEPROM+$5555
        rts                     ;writes are now enabled

;***************************************************************
;
;  FlashDelay
;
;  This function performs a 200 us delay (assuming 40MHz clock).
;  It is used to wait for the programming cycle to begin.
;
;***************************************************************
FlashDelay
        rep     #FLASH_DELAY
        nop
        rts

;***************************************************************
;
;  FlashCheck
;
;  This function waits for the programming cycle to complete.
;  It uses the EEPROM toggle bit method.
;
;***************************************************************
FlashCheck
        btst    #TOGGLE_BIT,x:EEPROM    ;check the initial status of bit 6
        jcs     check_set
check_clr
;        jsr     FlashDelay
        btst    #TOGGLE_BIT,x:EEPROM    ;test bit, it was clear
        jcs     check_set               ;if it is now set, continue...
        rts
check_set
;        jsr     FlashDelay
        btst    #TOGGLE_BIT,x:EEPROM    ;test bit, it was set
        jcc     check_clr               ;if it is now clear, continue...
        rts

;***************************************************************
;
;  FlashInit
;
;  This function performs the initialization required before
;  the flash routines are used.
;
;***************************************************************
FlashInit
        movep   #INIT_PLL,x:PLL ;set PLL
        movep   #INIT_BCR,x:BCR ;set BCR
        ori     #3,mr           ;disable interrupts
        movec   #0,sp           ;reset stack pointer
        movec   #0,omr          ;single chip mode
        nop                     ;delay for remapping
        jmp     (r0)            ;return

;***************************************************************
;
;  WriteWord
;
;  This function writes a word to the EEPROM one byte at a
;  time, starting with the low byte.
;
;***************************************************************
WriteWord
        move    a1,p:(r1)+      ;move low byte to FLASH EEPROM
        rep     #8
        lsr     a
        move    a1,p:(r1)+      ;move mid byte to FLASH EEPROM
        rep     #8
        lsr     a
        move    a1,p:(r1)+      ;move high byte to FLASH EEPROM
        rts

;***************************************************************
;
;  FlashBoot
;
;  This function executes the code to enter the DSP
;  bootstrap mode (Mode 1).
;
;***************************************************************
FlashBoot
        ori     #3,mr           ;disable interrupts
        movec   #0,sp           ;reset stack pointer
        movec   #1,omr          ;bootstrap mode
        nop                     ;delay for remapping
        jmp     <$0             ;begin bootstrap



BOOT_LOAD
        org     p:BOOT_RUN,p:

;***************************************************************
;
;  Boot
;
;  This is the boot routine. It is an overlay that is copied
;  from its P memory load address (EEPROM @ P:$C000) to its
;  P memory runtime address (RAM @ P:$0000) and executed by
;  the DSP bootloader. It then copies the loader routine from
;  its P memory load address to its P memory runtime address
;  and then jumps to it.
;
;***************************************************************
Boot
        jmp     DoBoot

        org     p:@LCV(R)+$40,p:@LCV(L)+$40

DoBoot
        move    #LOAD_LOAD,r0   ;get loader load address
        move    #LOAD_RUN,r1    ;get loader runtime address

        do      #LOAD_LENGTH,LoadLoop
        move    p:(r0)+,x0      ;read code from load address
        move    x0,p:(r1)+      ;write code to runtime address
LoadLoop

        jmp     Load            ;jump to the loader routine
        rts

LOAD_LOAD
        org     p:LOAD_RUN,p:

;***************************************************************
;
;  Load
;
;  This is the loader routine. It is an overlay that the boot
;  routine copies from its load address to its runtime address
;  and then executes. This routine then loads P, X, and Y RAM
;  from the P, X, and Y EEPROM, and then jumps to P:$0000 to
;  execute the application.
;
;***************************************************************
Load

LoadP
        move    #RAM_P_INT,r0           ;get start address of P RAM
        move    #EEPROM_P_INT,r1        ;get start address of P EEPROM
        move    #BLOCKS_P_INT*64,r2     ;get number of words to copy
        jsr     BootCopyP               ;Load P RAM from EEPROM
        move    #RAM_P_EXT,r0           ;get start address of P RAM
        move    #EEPROM_P_EXT,r1        ;get start address of P EEPROM
        move    #BLOCKS_P_EXT*64,r2     ;get number of words to copy
        jsr     BootCopyP               ;Load P RAM from EEPROM

LoadX
        move    #RAM_X_INT,r0           ;get start address of X RAM
        move    #EEPROM_X_INT,r1        ;get start address of X EEPROM
        move    #BLOCKS_X_INT*64,r2     ;get number of words to copy
        jsr     BootCopyX               ;Load X RAM from EEPROM
        move    #RAM_X_EXT,r0           ;get start address of X RAM
        move    #EEPROM_X_EXT,r1        ;get start address of X EEPROM
        move    #BLOCKS_X_EXT*64,r2     ;get number of words to copy
        jsr     BootCopyX               ;Load X RAM from EEPROM

LoadY
        move    #RAM_Y_INT,r0           ;get start address of Y RAM
        move    #EEPROM_Y_INT,r1        ;get start address of Y EEPROM
        move    #BLOCKS_Y_INT*64,r2     ;get number of words to copy
        jsr     BootCopyY               ;Load Y RAM from EEPROM
        move    #RAM_Y_EXT,r0           ;get start address of Y RAM
        move    #EEPROM_Y_EXT,r1        ;get start address of Y EEPROM
        move    #BLOCKS_Y_EXT*64,r2     ;get number of words to copy
        jsr     BootCopyY               ;Load Y RAM from EEPROM

LoadEnd
        jmp     START           ;run the application

;***************************************************************
;
; BootCopyP
;
; This function copies from the FLASH EEPROM to the P RAM.
;
; Inputs:
;
;       r0 : Starting address of the RAM for writing
;       r1 : Starting address of the FLASH EEPROM for reading
;       r2 : Number of words to copy
;
; Outputs:
;
; Registers modified:
;
;       a, r0, r1
;
;***************************************************************
BootCopyP
        do      r2,BootLoopP1   ;loop for number of words
        do      #3,BootLoopP2   ;loop to read 3 bytes and pack into a word
        move    p:(r1)+,a2      ;read byte from FLASH EEPROM
        rep     #8              ;shift right eight bits
        asr     a
BootLoopP2
        move    a1,p:(r0)+      ;write word to RAM
BootLoopP1
        nop
        rts

;***************************************************************
;
; BootCopyX
;
; This function copies from the FLASH EEPROM to the X RAM.
;
; Inputs:
;
;       r0 : Starting address of the RAM for writing
;       r1 : Starting address of the FLASH EEPROM for reading
;       r2 : Number of words to copy
;
; Outputs:
;
; Registers modified:
;
;       a, r0, r1
;
;***************************************************************
BootCopyX
        do      r2,BootLoopX1   ;loop for number of words
        do      #3,BootLoopX2   ;loop to read 3 bytes and pack into a word
        move    p:(r1)+,a2      ;read byte from FLASH EEPROM
        rep     #8              ;shift right eight bits
        asr     a
BootLoopX2
        move    a1,x:(r0)+      ;write word to RAM
BootLoopX1
        nop
        rts

;***************************************************************
;
; BootCopyY
;
; This function copies from the FLASH EEPROM to the Y RAM.
;
; Inputs:
;
;       r0 : Starting address of the RAM for writing
;       r1 : Starting address of the FLASH EEPROM for reading
;       r2 : Number of words to copy
;
; Outputs:
;
; Registers modified:
;
;       a, r0, r1
;
;***************************************************************
BootCopyY
        do      r2,BootLoopY1   ;loop for number of words
        do      #3,BootLoopY2   ;loop to read 3 bytes and pack into a word
        move    p:(r1)+,a2      ;read byte from FLASH EEPROM
        rep     #8              ;shift right eight bits
        asr     a
BootLoopY2
        move    a1,y:(r0)+      ;write word to RAM
BootLoopY1
        nop
        rts

LOAD_LENGTH     equ     *-Load

        end


